# PSR-7 Message Implementation

This repository contains a full [PSR-7](http://www.php-fig.org/psr/psr-7/)
message implementation, several stream decorators, and some helpful
functionality like query string parsing.


[![Build Status](https://travis-ci.org/guzzle/psr7.svg?branch=master)](https://travis-ci.org/guzzle/psr7)


# Stream implementation

This package comes with a number of stream implementations and stream
decorators.


## AppendStream

`GuzzleHttp\Psr7\AppendStream`

Reads from multiple streams, one after the other.

```php
use GuzzleHttp\Psr7;

$a = Psr7\Utils::streamFor('abc, ');
$b = Psr7\Utils::streamFor('123.');
$composed = new Psr7\AppendStream([$a, $b]);

$composed->addStream(Psr7\Utils::streamFor(' Above all listen to me'));

echo $composed; // abc, 123. Above all listen to me.
```


## BufferStream

`GuzzleHttp\Psr7\BufferStream`

Provides a buffer stream that can be written to fill a buffer, and read
from to remove bytes from the buffer.

This stream returns a "hwm" metadata value that tells upstream consumers
what the configured high water mark of the stream is, or the maximum
preferred size of the buffer.

```php
use GuzzleHttp\Psr7;

// When more than 1024 bytes are in the buffer, it will begin returning
// false to writes. This is an indication that writers should slow down.
$buffer = new Psr7\BufferStream(1024);
```


## CachingStream

The CachingStream is used to allow seeking over previously read bytes on
non-seekable streams. This can be useful when transferring a non-seekable
entity body fails due to needing to rewind the stream (for example, resulting
from a redirect). Data that is read from the remote stream will be buffered in
a PHP temp stream so that previously read bytes are cached first in memory,
then on disk.

```php
use GuzzleHttp\Psr7;

$original = Psr7\Utils::streamFor(fopen('http://www.google.com', 'r'));
$stream = new Psr7\CachingStream($original);

$stream->read(1024);
echo $stream->tell();
// 1024

$stream->seek(0);
echo $stream->tell();
// 0
```


## DroppingStream

`GuzzleHttp\Psr7\DroppingStream`

Stream decorator that begins dropping data once the size of the underlying
stream becomes too full.

```php
use GuzzleHttp\Psr7;

// Create an empty stream
$stream = Psr7\Utils::streamFor();

// Start dropping data when the stream has more than 10 bytes
$dropping = new Psr7\DroppingStream($stream, 10);

$dropping->write('01234567890123456789');
echo $stream; // 0123456789
```


## FnStream

`GuzzleHttp\Psr7\FnStream`

Compose stream implementations based on a hash of functions.

Allows for easy testing and extension of a provided stream without needing
to create a concrete class for a simple extension point.

```php

use GuzzleHttp\Psr7;

$stream = Psr7\Utils::streamFor('hi');
$fnStream = Psr7\FnStream::decorate($stream, [
    'rewind' => function () use ($stream) {
        echo 'About to rewind - ';
        $stream->rewind();
        echo 'rewound!';
    }
]);

$fnStream->rewind();
// Outputs: About to rewind - rewound!
```


## InflateStream

`GuzzleHttp\Psr7\InflateStream`

Uses PHP's zlib.inflate filter to inflate deflate or gzipped content.

This stream decorator skips the first 10 bytes of the given stream to remove
the gzip header, converts the provided stream to a PHP stream resource,
then appends the zlib.inflate filter. The stream is then converted back
to a Guzzle stream resource to be used as a Guzzle stream.


## LazyOpenStream

`GuzzleHttp\Psr7\LazyOpenStream`

Lazily reads or writes to a file that is opened only after an IO operation
take place on the stream.

```php
use GuzzleHttp\Psr7;

$stream = new Psr7\LazyOpenStream('/path/to/file', 'r');
// The file has not yet been opened...

echo $stream->read(10);
// The file is opened and read from only when needed.
```


## LimitStream

`GuzzleHttp\Psr7\LimitStream`

LimitStream can be used to read a subset or slice of an existing stream object.
This can be useful for breaking a large file into smaller pieces to be sent in
chunks (e.g. Amazon S3's multipart upload API).

```php
use GuzzleHttp\Psr7;

$original = Psr7\Utils::streamFor(fopen('/tmp/test.txt', 'r+'));
echo $original->getSize();
// >>> 1048576

// Limit the size of the body to 1024 bytes and start reading from byte 2048
$stream = new Psr7\LimitStream($original, 1024, 2048);
echo $stream->getSize();
// >>> 1024
echo $stream->tell();
// >>> 0
```


## MultipartStream

`GuzzleHttp\Psr7\MultipartStream`

Stream that when read returns bytes for a streaming multipart or
multipart/form-data stream.


## NoSeekStream

`GuzzleHttp\Psr7\NoSeekStream`

NoSeekStream wraps a stream and does not allow seeking.

```php
use GuzzleHttp\Psr7;

$original = Psr7\Utils::streamFor('foo');
$noSeek = new Psr7\NoSeekStream($original);

echo $noSeek->read(3);
// foo
var_export($noSeek->isSeekable());
// false
$noSeek->seek(0);
var_export($noSeek->read(3));
// NULL
```


## PumpStream

`GuzzleHttp\Psr7\PumpStream`

Provides a read only stream that pumps data from a PHP callable.

When invoking the provided callable, the PumpStream will pass the amount of
data requested to read to the callable. The callable can choose to ignore
this value and return fewer or more bytes than requested. Any extra data
returned by the provided callable is buffered internally until drained using
the read() function of the PumpStream. The provided callable MUST return
false when there is no more data to read.


## Implementing stream decorators

Creating a stream decorator is very easy thanks to the
`GuzzleHttp\Psr7\StreamDecoratorTrait`. This trait provides methods that
implement `Psr\Http\Message\StreamInterface` by proxying to an underlying
stream. Just `use` the `StreamDecoratorTrait` and implement your custom
methods.

For example, let's say we wanted to call a specific function each time the last
byte is read from a stream. This could be implemented by overriding the
`read()` method.

```php
use Psr\Http\Message\StreamInterface;
use GuzzleHttp\Psr7\StreamDecoratorTrait;

class EofCallbackStream implements StreamInterface
{
    use StreamDecoratorTrait;

    private $callback;

    public function __construct(StreamInterface $stream, callable $cb)
    {
        $this->stream = $stream;
        $this->callback = $cb;
    }

    public function read($length)
    {
        $result = $this->stream->read($length);

        // Invoke the callback when EOF is hit.
        if ($this->eof()) {
            call_user_func($this->callback);
        }

        return $result;
    }
}
```

This decorator could be added to any existing stream and used like so:

```php
use GuzzleHttp\Psr7;

$original = Psr7\Utils::streamFor('foo');

$eofStream = new EofCallbackStream($original, function () {
    echo 'EOF!';
});

$eofStream->read(2);
$eofStream->read(1);
// echoes "EOF!"
$eofStream->seek(0);
$eofStream->read(3);
// echoes "EOF!"
```


## PHP StreamWrapper

You can use the `GuzzleHttp\Psr7\StreamWrapper` class if you need to use a
PSR-7 stream as a PHP stream resource.

Use the `GuzzleHttp\Psr7\StreamWrapper::getResource()` method to create a PHP
stream from a PSR-7 stream.

```php
use GuzzleHttp\Psr7\StreamWrapper;

$stream = GuzzleHttp\Psr7\Utils::streamFor('hello!');
$resource = StreamWrapper::getResource($stream);
echo fread($resource, 6); // outputs hello!
```


# Static API

There are various static methods available under the `GuzzleHttp\Psr7` namespace.


## `GuzzleHttp\Psr7\Message::toString`

`public static function toString(MessageInterface $message): string`

Returns the string representation of an HTTP message.

```php
$request = new GuzzleHttp\Psr7\Request('GET', 'http://example.com');
echo GuzzleHttp\Psr7\Message::toString($request);
```


## `GuzzleHttp\Psr7\Message::bodySummary`

`public static function bodySummary(MessageInterface $message, int $truncateAt = 120): string|null`

Get a short summary of the message body.

Will return `null` if the response is not printable.


## `GuzzleHttp\Psr7\Message::rewindBody`

`public static function rewindBody(MessageInterface $message): void`

Attempts to rewind a message body and throws an exception on failure.

The body of the message will only be rewound if a call to `tell()`
returns a value other than `0`.


## `GuzzleHttp\Psr7\Message::parseMessage`

`public static function parseMessage(string $message): array`

Parses an HTTP message into an associative array.

The array contains the "start-line" key containing the start line of
the message, "headers" key containing an associative array of header
array values, and a "body" key containing the body of the message.


## `GuzzleHttp\Psr7\Message::parseRequestUri`

`public static function parseRequestUri(string $path, array $headers): string`

Constructs a URI for an HTTP request message.


## `GuzzleHttp\Psr7\Message::parseRequest`

`public static function parseRequest(string $message): Request`

Parses a request message string into a request object.


## `GuzzleHttp\Psr7\Message::parseResponse`

`public static function parseResponse(string $message): Response`

Parses a response message string into a response object.


## `GuzzleHttp\Psr7\Header::parse`

`public static function parse(string|array $header): array`

Parse an array of header values containing ";" separated data into an
array of associative arrays representing the header key value pair data
of the header. When a parameter does not contain a value, but just
contains a key, this function will inject a key with a '' string value.


## `GuzzleHttp\Psr7\Header::normalize`

`public static function normalize(string|array $header): array`

Converts an array of header values that may contain comma separated
headers into an array of headers with no comma separated values.


## `GuzzleHttp\Psr7\Query::parse`

`public static function parse(string $str, int|bool $urlEncoding = true): array`

Parse a query string into an associative array.

If multiple values are found for the same key, the value of that key
value pair will become an array. This function does not parse nested
PHP style arrays into an associative array (e.g., `foo[a]=1&foo[b]=2`
will be parsed into `['foo[a]' => '1', 'foo[b]' => '2'])`.


## `GuzzleHttp\Psr7\Query::build`

`public static function build(array $params, int|false $encoding = PHP_QUERY_RFC3986): string`

Build a query string from an array of key value pairs.

This function can use the return value of `parse()` to build a query
string. This function does not modify the provided keys when an array is
encountered (like `http_build_query()` would).


## `GuzzleHttp\Psr7\Utils::caselessRemove`

`public static function caselessRemove(iterable<string> $keys, $keys, array $data): array`

Remove the items given by the keys, case insensitively from the data.


## `GuzzleHttp\Psr7\Utils::copyToStream`

`public static function copyToStream(StreamInterface $source, StreamInterface $dest, int $maxLen = -1): void`

Copy the contents of a stream into another stream until the given number
of bytes have been read.


## `GuzzleHttp\Psr7\Utils::copyToString`

`public static function copyToString(StreamInterface $stream, int $maxLen = -1): string`

Copy the contents of a stream into a string until the given number of
bytes have been read.


## `GuzzleHttp\Psr7\Utils::hash`

`public static function hash(StreamInterface $stream, string $algo, bool $rawOutput = false): string`

Calculate a hash of a stream.

This method reads the entire stream to calculate a rolling hash, based on
PHP's `hash_init` functions.


## `GuzzleHttp\Psr7\Utils::modifyRequest`

`public static function modifyRequest(RequestInterface $request, array $changes): RequestInterface`

Clone and modify a request with the given changes.

This method is useful for reducing the number of clones needed to mutate
a message.

- method: (string) Changes the HTTP method.
- set_headers: (array) Sets the given headers.
- remove_headers: (array) Remove the given headers.
- body: (mixed) Sets the given body.
- uri: (UriInterface) Set the URI.
- query: (string) Set the query string value of the URI.
- version: (string) Set the protocol version.


## `GuzzleHttp\Psr7\Utils::readLine`

`public static function readLine(StreamInterface $stream, int $maxLength = null): string`

Read a line from the stream up to the maximum allowed buffer     ;kw8	6q,59gelzdY'Nm3lIʖo/ۺFsE<Ӌ0MÄ{u|pblku9"
Yq)`9Eu#gZbARM2sNN S&gyLi慭ќ)Mg<ᯏ:='O?>j)	Ynk9#4s˄s<gYco(9ٓ=<qf.%3ٿ&mI %6N]
}x>r߯/߽/oS4D;ŝ͉vk2x:'I|MqHtYx$K#2?&(a,myν$SD4Ƭψx-͙,{:*èGs2cs,4˭{O^}}lE,;Y{P(fN%?ĸF8+`FndO";C$KfMJ[6%B*0WwFgA<gt"fh[]ALϤMu\Y,-RZA_7m zZj{t9;RA6LR$QwͻI̍B]Yt' W+L 
՝^l.nZnZԻ.V{eeE<-:@X_Ց(A@f4b$:'g[̀.Ef
[klw	|s|wt։?2F1 sL8>4㹞|$;[&:t2C3/ޅdzQ`'ڧ&qh}" @]x#cՁHV<r]ZC PjiFO;z3ץz`
.k滁CLg9Wl4n4ROUSq7Z]:\4[G([VF@gŋߎsv:[q2h!qIWΛyL}ۯ	-_>m:wh~4Wﯓ?>Z%0)l{J}P7ڹJAb杺a8enRDem}.͜]6@2^(E4\m٧ -nZȳ(I l$gd#gBlotDW4vrV|&[;䌓g.hO=)N]ݤ:=؜s̈Mđ汤a	HR+
;݈V041XcHi-y .m8T|Yyt9uXLJ)@׳A(6T(!z1W҈xȭQBM(92E=\+H`,٭/j?{~G~gLr3'P{49g&!҆qS5{zy7dlor$@?t'PܪzqlٛW-KHe`)1Z,s#FPdBk C.jPݮ\Ms!ADip2r
h?l°H̙88	97+>%}xXeM~k7[%8tTZq|fLW+}I̐U0ɜ@7R{`Bꪖ#
[|#tXrJt>ATjH;qfH+փ3'h#1J`fpb":cjoG@
L
lшᚶ *6Ī
 *"6oδ2$_Տ8]>Í@3yilK5@oZ/ifGE=ͰkV_Q>Z}jH4r$Y/.^iab4
0UPXX<*Ms,v#񭆀0-exdR^RiC\pt/.^714.*\#߁Be^4m0"	##oOr`0աBi۾(A!xAsW
IL7Tj'H,܃
p_7?4pzlܜp:6SJeMkx6`֖T_gKbMDz	I%V\)K{̹y%wdY)	naN6|a۾uڻ6nCI(	_2!Ǟ+2!Y qkg]Dj%* GJ[#ф܍wd6Rz&p쮳*uI4370]A,:M(\aXujpS7 V*|}<1#V&i\cr;guϺ]P,{y嫴4bgSe u
"l>HbFj7^hjCG 3(("($s3<Xv`h2t!>}db,6c;,J v	I<PʞjxkaYMUSFI9d"U]wūM	tC2>s8IY~fh7^%'C)0iHeEw\DwܗT&)\Ms6xךK}`xF+]pAKXcMv 
D)U
pC|jAݡC+ʷT>LKf%^"+o2D&Jasv^VyVTŏGU#d֎@v|dAcm.PJd[Liȥ,W䁋JRcyb5sʮJmd9;:<dHhja6 %Aַi]9VK<X\/(V9)y Bﭦp3+f'):v@:@=|y,ߑwwX|Y 8+&t7JwP*5l2s#bDMʻln jצv
	V%20eeHڧit,r02f~rMjKF?j~z8_9sp%81Jx<;:.ܛR<LWrwdTFU-ˤ2P#0fftR|)mzT3ÈⅢBXdH`P^UDvx rE|+<F_YAڟVs7HhVVOndQTUg@: Д3ߛmf
1B @b(?Nv50j^a3&vWbV6޵,PX>*~U JE⽊īېkK'X1|u<0\AKYB֯mTf8	!*k1&rWSdT=ME:m8AcA)0QcU(SM:AYp*ZoQ
=<aT{Ͱ	?gaUw24nD@ty7NV ;\!=:`tÉa
)_6?U+	"~ej<U߀ទ/爗DYe.O81<~SPJ@QP	-R=w0Dͼ`>9Oy9yCqۍG,/Ϸ51ŗ=ֿ?!eUsYWػ]ԩ5cyNBfUx@Byn  wFWZ?QْIzJZ?o\@D5\o>/:xu8%ۋɣ39Paglv}qVsEeG`@g#> N{_ʛg!}???<7{=,M4Ü6~1_i>e?ƞX1_=CWF@	[Tҷ佾9[}<HUMxNnaC1sS5I@_ͬ)2eșW3{<T!	T	g	Iӵf}`W+JVM(nXʏ,OeD.<%<
=B"KsR,%k 4i:<Sm]zVF$`7qxzUW+ȱKMwA<҉ΌdSaj&Ƽ]zB<,s,!0[7VImlG!^ٜ>a:[s<5l15E=ύ~r3~h^*VGQ?>d㓓_'ox4ݟ.K|zL-Q7d'7|''|nkyZzJzƵq+E2ϽZť~fUm!154qcɢ.
e
9mWj1b6rZ@HK)3,ׁ\\
_Եf-;jz8ލi[מQu+ C9$#]JȨ
%X< Y|Wyχ|r0rCi֎#ȾShhi|:C\~˛u.?\ޯl*7[m#rMۊjF IWA0lnPɡ{/я϶SROAWrZoD{&"V|-|;ҀA~l-
srxQ+ReβZH^uc~Es4_j*JQq>)dO.em2/ΗQM/?C[e|AչT0KTW߆-,rIQE'40$.o
 nr?p+c*#8CJ1@%!6zcK|mjgt+pϷF˿150u?h
'_`DBGjrDgO[xl^7_mlQ\!P(k(RE;CuG Ub дzkTBR[i	kg~|㝕ł(N|c: UAYtpAMRJ94S^íV
e;a1QF_?=U|O
G\p c8?pzNn^@	F(U,b$%xL!yͮF.832``	g #|Fg$so'?}'OJ&}LkӋu0
4\6
aqozHU'# ϘL+#9`roG0䵃-fw	8̚U<8Pd24ed-~߿hÜCa
&!MPXx$9;mL
Y.`6Qk؎vÏ#=QWAh	{xV3kΩ^
Y~OFO'h$Ta`=Sy*͑b8}~W؃lDzyOqbsh"xnAAo^@/zH}!a=:X,79OX$(C29ƽ!O(+)D
r8'cˆgKM/Xv4V<2'f   =WLӿ<}=D((Z+Vk-oOM^P߿kBh|9d7{ZwEӠawo4:)bpB|2\zիWտ^aՙ`iABtEt-ջj ]{0"Gbs#'q׵4)(uG]`J`9YBWSmBTY#KHK0oxʿ S95Z/6jBma|˼%]_5K<iNR!tk	SGoz4	-WU`rHj|%Zkތ7+h2vSg</sk陨cE OtU՟oF
7yh&ZC1SS_qTI>٥l^#{4͆+(뱂><0lVϡ0|6Y9i|'6nVǕn˖ő=`ampf?L=F>l@2>}cdITx!wB,4lD֞pZ@̄5H[p|}Q6tHD,<T0s&ټsç_nȗJ+& N>!f?~k/R¼yB?j3mLfks910wyHT^@%3 qpwMtdrMcVΚ`nr|Teӧ1tb4Yz;LV"	\ q7ß ԙ];fFCx8A~!2m(}z)aS(71uӐ3G2XDĮ)miuMnfpK<ulR(jwOkWCJm*/	/GQ{ԐFLfRa¾p	BO OHC7)Xؤ*$jZHBimjN\:k2 /,o_Zdn4¸LS2"HKT'LJ&p5ǈ 7S *dl[_qyמl>#D%\E +جc
x{a蓮R8}Qk*1R) 49NDH$e} ʁ^STHU[[p&Zy"3iyvl
:TG|sx 98cǃxQ;ɴlяxz0Gy-}	i<q ĸ`>?e֗Y<'#v2쬽X-%{;RjBk0VY+#tw9e( A 4Fvȭ ڌj0] $(M)v}H'mxPp% @"!'&/bfg@Sg%WvԹt',ձ\1KKK9F&ٰщ|1>fIq0"=pܿ׷hBBTwݙߟ8ZDU)kLr EۖZWSS~v9L!
(h5x6_Hd|g==*I{իk$³qْ )<1,v_,\tM*LũPUse/N3om4͟##2WVxW{0oE{͟Kag`zeVAi[.<dVS^5]&
uIX<2'cMdH)FDj&
 +~Q2$9><VJ>貦xzŕGDq%x$$@$ݾtycx?R+IF$F.3	Hb"H,cgm!2Mx*@_>4wҩہ(ZIy(5>.Χf)>$	N/d;ahy~"!kWvr	Wg/4Bʅ7'h⒭H6/Jd-.`b?zveq?Q'Y[hB:*Lv-j
H& ٤Dm'bßإ;7i-6V<Y/ѪV)\9_iry	tjjrHUbz*QޟBb_4Kmjn8idF"`:u³P=jGSr& S.5<M?Z"BigpN-	)-Zn,q`}x
lU33VƟF~msMF ڥ46ɐQT_iWmYη/;T5]h7ӦBw/ZP&2c<8Ѷ/#ӹ|A e\sX
e
.
^BsL̫
5=Y`hTإ싹^LbZHפf	Zj^iɶ	ll`gxXiRshX?UBoijS#n(|4ƌ1]UcSZYl<dRn]~]$|ߴ[moP Uq6wx0rkf:L,O! D1}yy҈Ku`jl ğ%@afdL]h-ՒĽK%#nLfe4%6..@"e^Oe\"׳0.S@ܺd7k>UGhY1FD WQ)=FzjVcJQ=l"|yeA0ƚ6?"bnfJlNAi|zOR=p֜%yO2/3ȶڧ)e^o*YRjG$=bIa7Őjn}O)3cq3*kڡ+L[:B%X<=Wn@PkXaמې=o6hGbKSqܚV݋~jݢ~5Ol`.#_	ʦJR_B^E,&Eu[1deaeue[O~[ٷyͅU-gO|7St
ޮ{&ndq;v@jv熋{%T!T97+f
&2{|2!20kn,pxO=2Xc_iC0kK(uk%{K%уլCg+	텢dEZIçʹEzA5BtR~"	~ϓqӞ}ڄ
%pv܌Ǳ3	JɈ-'["
Eq=uHԔo5D\j0Ű՗U,Uܲ*dh3l2f),K~IؔUXEVXXƪ^"<CmnߊAxJW~'tw;"*1'ܾ '6Gʧy}pZ9ZchfJbt
?g²E]^tJWKμ1`{!#dGtpD7F'[2
QY5JAHД^	}1BToG}cH
tPRvQ`}lD{=!>9΃M3~Y\,!`Nө2J G ,Bε&0(3+*'I4& I]_Ϊ
hqfr,(Kc@@<Y
p.Ks];T^
+[d6d)KOw'k)[cV:ɉզ]ImM(J02`gz̷@t)ާq@U+j}J[SsGƶ1J~8
l4lWR?c+lV ~F)F!+֩Gxd.>d#84a_h˧'
5=<]'ya(#]AaJuQZ;"+?gLL3y-^pS;0#;}VfNGG*Sǭmicxo*:iY=Ğ>ՏGuTv}l}Rw֡JߩTcҝqH7{:yoJ+u<ntNux;?ChGm$oU1,ˣUvZ<GLOaBU|}BFit 	SiK;&F8;&GinPpWDʟމv,u1o*K"1G[>oWOYI$xbޙ1Y?*Swxޣ7r
86mG8ٴE75YO'MZN!?ÒPςƮ0d4O88UG]Ul|
~2B\	pZ|߃ 4Nx0J)A"FK\5&I#t< bY(kl48Tqx Wt_U!l>\d|*7)։^
MK+I<R 4aV#OW^1Р&zOC9+ ^47p|C_&PasSlbՄrXY_ǒ=B_ d|]TY^O"U>ռykQ^ތ=$R!]ZXl]D~ e0d|]z6M9@Z{ih</	O1
߀-dO 1^2Gj\9GN`kEG |@xطyIycַV*#ģHփ9naRPB	n'\Ywe$I(^NA07ź
b(X'SFf>kP~,`#K3$f[-lg^
 mg
ʖLAĥ&*)G
:ՔE88i &o\Y /⋧c/`
_ׯx*Q*)8ZTXo"[P3)
*r?b]z>~Sm\hӏ{I
>M /{	Yw+FA[ʟ7zUlU'r|RT xu
UK=CfWj	ꅛԐA蓎t-ɀ
!SJG	iu)b{-9u1jU"t b^Mս<,LK=RX,0ڥ~H,Ζh7Uc#J¾2 P5t)tXf.~cC
{s疍W³)cG1TR0.0
#`DJ׳<"&67kX0-z|.[8"cu9^߄|
	/)nel9]==QA^iPcPh
K5w@uH	 0;r5B͒ *><	CScCO_N:Koz<bEN#<  =k{;+&sB܇Piڜ^
J=ҳg`ٖe[eɐɀSA҂kR-"	x5V&r|")ڦl9ۿ#w %%h$qX-TlpBͥU~ oPUǀv
I5hl<|
^OϏcobr^;jiCK젤dj4<F`xK#
I$JRJm+<u$EO34Pw`& =k+Tib^x\ lH"⻑rE ?ůN\oya֝1{5f0틁
M܊i[ʟ8:ךaE=WAÅ"(؟H|wAm`*!<踦&v]&]I#SU
VL\`Cf>`|b.9Oذ'hXw4«\yeh>2z8'rp!ټ%7Ͻr`7gX?p㚒qö;Sz$bƴϳr9)w26U\Jg ,d:86ώK;4\wb_%jox{o4m>o^uMӅ|$v5mM6mo7,8F&9Ma*]aB##7F[&:kZ)Y._VNЖk^	2&ʹ0eT;C>e7FPS]ڣs۹y#"O Z@
X>@8
=wL~6ogĭD$L&vS$KQeϑ![ZP,:?s%`pGŽ3pҠgTh=ٗ76x<Op
!Õ
!kV^@f
e9@9e4=/
>P0fQ7#3<qΰgϭȚA=:6K8?B*u	#9W⪃Gݡ~Zn:5Ϸ3rot0lEVb.޶ogBmC/bY>ZtaT4LvshĞ(gOKfSHR̓i:۰Lr

]GsW(ǔAShi:j={0ͳv^s7֡G>Lq>e{xzVjqrd]r}za^N|, #G v&XsL.y,lgOMzn}ݾnQg<䋩!_L':ˇ
[DYG
Xvܙ~-票@5)QeMDuYaL1M8>Sڹ^>!?Px^o>InRJm|\!t3B#ag=29Q?5>?jfJx ܶmZGj]$4+CT!YTk-=8~.t	:֮'zs2w'eO&"\fr}r*VYMѲg|z`.UV#Oq^0^Ճ5szu7zlF/>)ڞ_ӶEeV-:K%Э~ccKGK6ϕlOn0jc Kl}BPg$jLQ4GWs1b/f`+6EJVt5r&55#,M#vԷX)RpO.:`_)"4
Sō(ϻj_s4/T }
-[/3/3<zrh?OQ7,La^
G5iQapTqR _xiuea8}%wxosD[Ж󅅕UMhM#l<sR)bSZ+F%{
̔jEnhnQxS\	l3 9E	^謹_ܾ@DP[* Pg	JMۢ?oG$
G=o\agЉNxϳ$	_WG-֡+L+!^Z.1&0~ד5E[f.DO")n$$HB~eyͷz@kKۑܒRs(Ӏt0ah@xAm,oapG݂@
1xJk7pQ6Θo#W7膋	}"M[啢(a5^}=PFHaO1| 	X1]ީ/ib
6ĕwǨ<vǕEt<5,%+$@b('qns@?8
֨+
D