<?php

namespace AMLA\SqrBundle\Controller;

use AMLA\SqrBundle\Entity\Repertoire;
use AMLA\SqrBundle\Entity\AccessInBill;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\JsonResponse;

/**
 * Repertoire controller.
 *
 * @Route("repertoire/{_locale}", defaults = {"_locale" = "fr"})
 */
class RepertoireController extends Controller
{
    /**
     * Lists all repertoire entities.
     *
     * @Route("/", name="repertoire_index")
     * @Method("GET")
     */
    public function indexAction()
    {
        if($this->getUser() && !empty($this->getUser()->getRole())) 
        {
            $fournisseur = $this->getUser()->getCompagnyId();
            $em = $this->getDoctrine()->getManager();

            $repertoires = $em->getRepository('AMLASqrBundle:Repertoire')->findBy(['fournisseur'=>$fournisseur]);

            return $this->render('repertoire/index.html.twig', array(
                'repertoires' => $repertoires,
            ));

        }else
        {
            $this->get('session')->getFlashBag()->add('danger', 'Désolé : vous n\'avez pas accès à cette ressource');
            return $this->redirectToRoute('dashboard_index');
        }
    }

    /**
     * Creates a new repertoire entity.
     *
     * @Route("/new", name="repertoire_new")
     * @Method({"GET", "POST"})
     */
    public function newAction(Request $request)
    {
        if($this->getUser() && !empty($this->getUser()->getRole())) 
        {
            $fournisseur = $this->getUser()->getCompagnyId();

            $repertoire = new Repertoire();
            $form = $this->createForm('AMLA\SqrBundle\Form\RepertoireType', $repertoire);
            $form->handleRequest($request);

            if ($form->isSubmitted() && $form->isValid()) 
            {

                $repertoire->setFournisseur($fournisseur);

                $em = $this->getDoctrine()->getManager();
                $em->persist($repertoire);
                $em->flush();

               // return $this->redirectToRoute('repertoire_show', array('id' => $repertoire->getId()));
                 return $this->redirectToRoute('repertoire_index');
            }

            return $this->render('repertoire/new.html.twig', array(
                'repertoire' => $repertoire,
                'form' => $form->createView(),
            ));

        }else
        {
            $this->get('session')->getFlashBag()->add('danger', 'Désolé : vous n\'avez pas accès à cette ressource');
            return $this->redirectToRoute('dashboard_index');
        }
    }

    /**
     * Finds and displays a repertoire entity.
     *
     * @Route("/{id}", name="repertoire_show")
     * @Method("GET")
     */
    public function showAction(Repertoire $repertoire)
    {
        $deleteForm = $this->createDeleteForm($repertoire);

        return $this->render('repertoire/show.html.twig', array(
            'repertoire' => $repertoire,
            'delete_form' => $deleteForm->createView(),
        ));
    }

    /**
     * Displays a form to edit an existing repertoire entity.
     *
     * @Route("/{id}/edit", name="repertoire_edit")
     * @Method({"GET", "POST"})
     */
    public function editAction(Request $request, Repertoire $repertoire)
    {
        $deleteForm = $this->createDeleteForm($repertoire);
        $editForm = $this->createForm('AMLA\SqrBundle\Form\RepertoireType', $repertoire);
        $editForm->handleRequest($request);

        if ($editForm->isSubmitted() && $editForm->isValid()) {
            $this->getDoctrine()->getManager()->flush();

            return $this->redirectToRoute('repertoire_edit', array('id' => $repertoire->getId()));
        }

        return $this->render('repertoire/edit.html.twig', array(
            'repertoire' => $repertoire,
            'edit_form' => $editForm->createView(),
            'delete_form' => $deleteForm->createView(),
        ));
    }

    /**
     * Deletes a repertoire entity.
     *
     * @Route("/{id}", name="repertoire_delete")
     * @Method("DELETE")
     */
    public function deleteAction(Request $request, Repertoire $repertoire)
    {
        $form = $this->createDeleteForm($repertoire);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $em = $this->getDoctrine()->getManager();
            $em->remove($repertoire);
            $em->flush();
        }

        return $this->redirectToRoute('repertoire_index');
    }

    /**
     * Creates a form to delete a repertoire entity.
     *
     * @param Repertoire $repertoire The repertoire entity
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createDeleteForm(Repertoire $repertoire)
    {
        return $this->createFormBuilder()
            ->setAction($this->generateUrl('repertoire_delete', array('id' => $repertoire->getId())))
            ->setMethod('DELETE')
            ->getForm()
        ;
    }

    /**
     * Lists all villes entities.
     *
     * @Route("/json/view", name="repertoire_index_fournisseur")
     * @Method({"GET", "POST"})
     */
    public function getFournisseurRepertoire(Request $request)
    {
        $em = $this->getDoctrine()->getManager();
        $idFournisseur = $request->get('idFournisseur');

        $fournisseur = $em->getRepository('AMLASqrBundle:Fournissseur')->findOneBy(['id'=>$idFournisseur]);
        //dump($fournisseur);die;
        if($fournisseur)
        {
            $repertoires = $em->getRepository('AMLASqrBundle:Repertoire')->findBy(['fournisseur'=>$fournisseur]);
            $tabRetour = array();

            foreach ($repertoires as $repertoire) 
            {
                $tabRetour[] = 
                [
                   'id'=>$repertoire->getId(), 
                    'nomRaisonSociale'=>$repertoire->getNomRaisonSociale(),
                    'prenomSigle'=>(empty($repertoire->getPrenomSigle())) ? 'N/A' : $repertoire->getPrenomSigle(),
                    'niu'=>(empty($repertoire->getNiu())) ? 'N/A' : $repertoire->getNiu(),
                ];
            }

            $return = array(
                        'Etat' =>'SUCCESS' , 
                        'data' =>$tabRetour, 
                        'Message' =>"compte retrouvé",
                    );
        }else
        {
            $return = array(
                        'Etat' =>'FAILED' , 
                        'Message' =>"Le compte est introuvable",
                    );  
        }

        

        $ret = new JsonResponse($return);
        return $ret;        
    }


    /**
     * Lists all villes entities.
     *
     * @Route("/save/selection/param", name="repertoire_save_selction")
     * @Method({"GET", "POST"})
     */
    public function saveFournisseurSelection(Request $request)
    {
        $em = $this->getDoctrine()->getManager();
        $repertoireId = $request->get('repertoireId');

        $repertoire = $em->getRepository('AMLASqrBundle:Repertoire')->findOneBy(['id'=>$repertoireId]);
        
        if($repertoire)
        {           
            $accessInBill = new AccessInBill();
            $accessInBill->setRepertoire($repertoire);
            $accessInBill->setInBill($repertoire->getFournisseur()->getInBill());

            $em->persist($accessInBill);
            $em->flush();
            $return = array(
                        'Etat' =>'SUCCESS' , 
                        'Message' =>"Enregistré avec succès",
                    );
        }else
        {
            $return = array(
                        'Etat' =>'FAILED' , 
                        'Message' =>"erreur d'enregistrement",
                    );  
        }

        $ret = new JsonResponse($return);
        return $ret;        
    }
}
